"""Dishwasher open/close tasks."""
from abc import ABC

import numpy as np

from bigym.bigym_env import BiGymEnv, MAX_DISTANCE_FROM_TARGET
from bigym.envs.props.dishwasher import Dishwasher
from bigym.utils.physics_utils import distance

DISHWASHER_POS = np.array([1.2, 0, 0])
DISHWASHER_ROT = np.array([0, 0, -np.pi / 2])

TOLERANCE = 0.05


class _DishwasherEnv(BiGymEnv, ABC):
    """Base dishwasher environment."""

    _DEFAULT_ROBOT_POS = np.array([0, -0.8, 1])

    def _initialize_env(self):
        self.dishwasher: Dishwasher = Dishwasher(self._mojo)
        self.dishwasher.body.set_position(DISHWASHER_POS)
        self.dishwasher.body.set_euler(DISHWASHER_ROT)

    def _fail(self) -> bool:
        return (
            distance(self._robot.pelvis, self.dishwasher.body)
            > MAX_DISTANCE_FROM_TARGET
        )


class DishwasherClose(_DishwasherEnv):
    """Close dishwasher task."""

    def _get_task_privileged_obs_space(self):
        return {}

    def _get_task_privileged_obs(self):
        return {}

    def _success(self) -> bool:
        return np.allclose(self.dishwasher.get_state(), 0, atol=TOLERANCE)

    def _on_reset(self):
        self.dishwasher.set_state(door=1, bottom_tray=1, middle_tray=1)


class DishwasherCloseTrays(DishwasherClose):
    """Close dishwasher trays task."""

    def _success(self) -> bool:
        return np.allclose(self.dishwasher.get_state()[1:], 0, atol=TOLERANCE)


class DishwasherOpen(_DishwasherEnv):
    """Open dishwasher task."""

    def _get_task_privileged_obs_space(self):
        return {}

    def _get_task_privileged_obs(self):
        return {}

    def _success(self) -> bool:
        return np.allclose(self.dishwasher.get_state(), 1, atol=TOLERANCE)

    def _on_reset(self):
        self.dishwasher.set_state(door=0, bottom_tray=0, middle_tray=0)


class DishwasherOpenTrays(DishwasherClose):
    """Open dishwasher trays task."""

    def _success(self) -> bool:
        return np.allclose(self.dishwasher.get_state()[1:], 1, atol=TOLERANCE)

    def _on_reset(self):
        self.dishwasher.set_state(door=1, bottom_tray=0, middle_tray=0)
